// ==UserScript==
// @name         5ch オリジナルポップアップ用 （aタグ削除＆span置換）新
// @namespace    http://tampermonkey.net/
// @version      1.5
// @description  >>123や>>123-125、>123などを含むaタグを削除し、span要素に置換（動的対応）
// @match        *://*.5ch.net/*
// @grant        none
// @run-at       document-end
// ==/UserScript==

(function() {
  'use strict';

  // 絶対URLに変換
  function resolveHref(href) {
    const a = document.createElement('a');
    a.href = href;
    return a.href;
  }

  // 数字範囲などからpostId配列を生成
  function extractPostNumbers(text) {
    const numbers = new Set();
    // >>5, >>5-10, >>5,10, >5 を許容する正規表現のマッチ文字列例
    // ここでは単純化のため '>>5-10'など全体を扱うが、後で分解
    const parts = text.replace(/^>+/, '').split(/,|、/);
    parts.forEach(part => {
      if (part.includes('-')) {
        let [startStr, endStr] = part.split('-');
        const start = parseInt(startStr, 10);
        let end = parseInt(endStr, 10);
        if (isNaN(end)) end = start + 50; // 未指定終端は +50件まで
        if (!isNaN(start)) {
          for (let i = start; i <= end && i <= start + 1000; i++) {
            numbers.add(i);
          }
        }
      } else {
        const num = parseInt(part, 10);
        if (!isNaN(num)) numbers.add(num);
      }
    });
    return Array.from(numbers).sort((a,b) => a-b);
  }

  // マッチテキストからspan要素作成
  function createSpan(text, baseUrl) {
    const postIds = extractPostNumbers(text);
    const span = document.createElement('span');
    span.textContent = text;
    if (postIds.length === 1) {
      span.className = 'custom_reply';
      span.dataset.postId = postIds[0];
      span.dataset.href = `${baseUrl}/${postIds[0]}`;
    } else {
      span.className = 'custom_reply';
      span.dataset.mentionpostids = JSON.stringify(postIds);
      span.dataset.href = `${baseUrl}/${text.replace(/^>+/, '')}`;
    }
    return span;
  }

  // テキストノード内の対象文字列を置換する
  function replaceTextNode(textNode, baseUrl) {
    const regex = />>?\d+(?:-\d+)?(?:[,、]\d+(?:-\d+)?)*/g;
    let text = textNode.textContent;
    let match, lastIndex = 0;
    const frag = document.createDocumentFragment();

    while ((match = regex.exec(text)) !== null) {
      // 置換前のテキストを追加
      frag.appendChild(document.createTextNode(text.slice(lastIndex, match.index)));
      // 置換用spanを追加
      frag.appendChild(createSpan(match[0], baseUrl));
      lastIndex = regex.lastIndex;
    }
    // 置換後の残りテキストを追加
    frag.appendChild(document.createTextNode(text.slice(lastIndex)));

    textNode.parentNode.replaceChild(frag, textNode);
  }

  // a.reply_linkをspan化し、さらにpost-content内のテキストも置換
  function processPostContent(postContent) {
    const baseUrl = location.origin + location.pathname.replace(/\/\d+$/, '');

    // a.reply_linkを完全削除し、テキスト部分は後で処理
    postContent.querySelectorAll('a.reply_link').forEach(a => {
      // aタグの中身テキストをaの親にテキストノードで戻す
      const text = a.textContent || '';
      const textNode = document.createTextNode(text);
      a.parentNode.replaceChild(textNode, a);
    });

    // テキストノードを置換してspan化
    const walker = document.createTreeWalker(postContent, NodeFilter.SHOW_TEXT);
    let node;
    const textNodes = [];
    while ((node = walker.nextNode())) {
      if (node.textContent.match(/>>?\d+/)) {
        textNodes.push(node);
      }
    }
    textNodes.forEach(textNode => replaceTextNode(textNode, baseUrl));
  }

  // 初回処理
  document.querySelectorAll('.post-content').forEach(processPostContent);

  // 動的監視
  const observer = new MutationObserver(mutations => {
    for (const mutation of mutations) {
      for (const node of mutation.addedNodes) {
        if (node.nodeType !== 1) continue;
        if (node.classList.contains('post-content')) {
          processPostContent(node);
        } else {
          node.querySelectorAll && node.querySelectorAll('.post-content').forEach(processPostContent);
        }
      }
    }
  });
  observer.observe(document.body, { childList: true, subtree: true });
})();
